"""
Plotly 시각화 모듈.

각 Figure 함수는 "입력 데이터프레임 → plotly Figure"를 생성한다.
학습 포인트:
- 시각화 함수는 "하나의 책임(한 가지 차트)"만 갖도록 분리한다.
"""

import pandas as pd
import plotly.express as px
from plotly.graph_objs import Figure


def fig_category_sales(filtered_frame: pd.DataFrame) -> Figure:
    """
    카테고리별 매출 합계를 막대 차트로 시각화한다.

    Parameters
    ----------
    filtered_frame : pandas.DataFrame
        필터가 적용된 데이터프레임.

    Returns
    -------
    plotly.graph_objs.Figure
        카테고리별 Sales 합계 막대 차트.
    """
    summary = (
        filtered_frame.groupby("Category", as_index=False)["Sales"]
        .sum()
        .sort_values("Sales", ascending=False)
    )
    figure = px.bar(summary, x="Category", y="Sales", title="카테고리별 매출")
    figure.update_layout(xaxis={"categoryorder": "total descending"}, margin=dict(l=20, r=20, t=40, b=20))
    return figure


def fig_channel_share(filtered_frame: pd.DataFrame) -> Figure:
    """
    채널별 매출 비중을 도넛 차트로 시각화한다.

    Parameters
    ----------
    filtered_frame : pandas.DataFrame
        필터가 적용된 데이터프레임.

    Returns
    -------
    plotly.graph_objs.Figure
        채널별 Sales 비중 도넛 차트.
    """
    summary = (
        filtered_frame.groupby("Channel", as_index=False)["Sales"]
        .sum()
        .sort_values("Sales", ascending=False)
    )
    figure = px.pie(summary, names="Channel", values="Sales", title="판매 방식별 매출 비중", hole=0.45)
    figure.update_layout(margin=dict(l=20, r=20, t=40, b=20))
    return figure


def fig_city_topn(filtered_frame: pd.DataFrame, top_n: int = 10) -> Figure:
    """
    도시별 상위 매출 Top-N을 막대 차트로 시각화한다.

    Parameters
    ----------
    filtered_frame : pandas.DataFrame
        필터가 적용된 데이터프레임.
    top_n : int, optional
        상위 도시 개수, 기본값 10.

    Returns
    -------
    plotly.graph_objs.Figure
        Top-N 도시별 Sales 막대 차트.
    """
    summary = (
        filtered_frame.groupby("City", as_index=False)["Sales"]
        .sum()
        .sort_values("Sales", ascending=False)
        .head(top_n)
    )
    figure = px.bar(summary, x="City", y="Sales", title=f"상위 {top_n} 도시별 매출")
    figure.update_layout(xaxis={"categoryorder": "total descending"}, margin=dict(l=20, r=20, t=40, b=20))
    return figure


def fig_price_qty_scatter(filtered_frame: pd.DataFrame) -> Figure:
    """
    가격(Price)과 수량(Quantity)의 관계를 산점도로 표현한다.

    - 점 색상: Category
    - Hover: City, Channel, Product

    Parameters
    ----------
    filtered_frame : pandas.DataFrame
        필터가 적용된 데이터프레임.

    Returns
    -------
    plotly.graph_objs.Figure
        가격-수량 산점도.
    """
    figure = px.scatter(
        filtered_frame,
        x="Price",
        y="Quantity",
        color="Category" if "Category" in filtered_frame.columns else None,
        hover_data=[c for c in ["City", "Channel", "Product"] if c in filtered_frame.columns],
        title="가격/수량 분포",
    )
    figure.update_traces(mode="markers", marker=dict(size=8, opacity=0.7))
    figure.update_layout(margin=dict(l=20, r=20, t=40, b=20))
    return figure


def fig_product_treemap(filtered_frame: pd.DataFrame, max_items: int = 5) -> Figure:
    """
    제품별 매출을 트리맵으로 시각화한다. (상위 max_items로 제한)

    Parameters
    ----------
    filtered_frame : pandas.DataFrame
        필터가 적용된 데이터프레임.
    max_items : int, optional
        트리맵에 표시할 (Sales 기준) 상위 아이템 수, 기본값 50.

    Returns
    -------
    plotly.graph_objs.Figure
        제품 트리맵.
    """
    summary = (
        filtered_frame.groupby(["Category", "Product"], as_index=False)["Sales"]
        .sum()
        .sort_values("Sales", ascending=False)
        .head(max_items)
    )

    # --- 방어적 변환: narwhals-like 객체를 pandas DataFrame으로 변환하고 인덱스 정리 ---
    try:
        if hasattr(summary, "to_pandas"):
            summary = summary.to_pandas()
        else:
            summary = pd.DataFrame(summary)
    except Exception:
        summary = pd.DataFrame(summary)

    # Sales 컬럼을 숫자로 강제하고 NaN은 0으로 채움
    if "Sales" in summary.columns:
        summary["Sales"] = pd.to_numeric(summary["Sales"], errors="coerce").fillna(0)
    else:
        summary["Sales"] = 0

    summary = summary.reset_index(drop=True)
    # -------------------------------------------------------------------------------

    # px.treemap 호출을 안전하게 감싸서 실패시 빈(플레이스홀더) Figure 반환
    try:
        figure = px.treemap(
            summary,
            path=["Category", "Product"],
            values="Sales",
            title=f"제품 매출 트리맵 (상위 {max_items}개)"
        )
        figure.update_layout(margin=dict(l=20, r=20, t=40, b=20))
    except Exception:
        import plotly.graph_objects as go
        figure = go.Figure().update_layout(title="데이터 오류 (데이터 없음)")

    return figure