"""
데이터 로딩 및 전처리 모듈.

- CSV를 읽어와 타입을 정규화한다.
- 결측값을 안전하게 보정한다.
- 학습을 위해 평균 단가 같은 파생 지표를 생성한다.

학습 포인트:
- "입력 데이터 정규화"는 시각화/모델링 품질에 직결된다.
- 데이터 스키마가 바뀌면 이 파일만 수정하면 된다.
"""

from pathlib import Path
import pandas as pd

# 프로젝트에서 기대하는 기본 컬럼 세트
NUMERIC_COLUMNS: list[str] = ["Price", "Quantity", "Sales"]
TEXT_COLUMNS: list[str] = ["City", "Channel", "Category", "Product"]


def load_sales_csv(csv_path: Path) -> pd.DataFrame:
    """
    sales_data.csv를 로드하고 기본 정규화를 수행한다.

    Parameters
    ----------
    csv_path : Path
        CSV 파일 경로.

    Returns
    -------
    pandas.DataFrame
        정규화된 데이터프레임. 다음을 보장한다:
        - 숫자형: Price, Quantity, Sales → 수치 변환 및 결측 0 대체
        - 텍스트: City, Channel, Category, Product → 결측 "Unknown"
        - 파생: AvgPrice_calc (Sales / Quantity, 0-division 안전 처리)
    """
    data_frame = pd.read_csv(csv_path)

    # 숫자형 컬럼: 안전한 수치 변환
    for column_name in NUMERIC_COLUMNS:
        if column_name in data_frame.columns:
            data_frame[column_name] = pd.to_numeric(data_frame[column_name], errors="coerce").fillna(0)

    # 텍스트 컬럼: 결측은 "Unknown"으로 보정
    for column_name in TEXT_COLUMNS:
        if column_name in data_frame.columns:
            data_frame[column_name] = data_frame[column_name].fillna("Unknown")

    # 평균 단가 파생 지표 (0으로 나누는 상황 방지)
    if {"Sales", "Quantity"}.issubset(data_frame.columns):
        data_frame["AvgPrice_calc"] = (data_frame["Sales"] / data_frame["Quantity"]).replace([float("inf")], 0).fillna(0)
    else:
        # Sales/Quantity가 없으면 Price를 대체 지표로 활용
        data_frame["AvgPrice_calc"] = data_frame.get("Price", 0)

    return data_frame


def unique_sorted(data_frame: pd.DataFrame, column_name: str) -> list[str]:
    """
    특정 컬럼의 유니크 값 목록을 정렬하여 반환한다.

    Parameters
    ----------
    data_frame : pandas.DataFrame
        원본 데이터프레임.

    column_name : str
        고유값을 구할 컬럼명.

    Returns
    -------
    list[str]
        정렬된 고유 문자열 목록. 컬럼이 없으면 빈 리스트.
    """
    if column_name not in data_frame.columns:
        return []
    return sorted(data_frame[column_name].dropna().astype(str).unique())
