"""
Dash 콜백 관리 모듈.

- 필터 변경 시 KPI와 모든 차트를 갱신한다.
- 데이터 필터링과 KPI 계산을 별도 함수로 분리하여 테스트 용이성을 높인다.

학습 포인트:
- 콜백 함수는 '입력 → 순수 계산 → 출력' 흐름으로 작성하면 디버깅이 쉽다.
"""

import pandas as pd
from dash import Dash, Input, Output
from services.layout import kpi_card
from services.figures import (
    fig_category_sales,
    fig_channel_share,
    fig_city_topn,
    fig_price_qty_scatter,
    fig_product_treemap,
)


def apply_filters(source_frame: pd.DataFrame, selected_cities: list[str] | None,
                  selected_channels: list[str] | None, selected_categories: list[str] | None) -> pd.DataFrame:
    """
    드롭다운에서 선택한 값들로 데이터프레임을 필터링한다.

    Parameters
    ----------
    source_frame : pandas.DataFrame
        원본 데이터프레임.
    selected_cities : list[str] | None
        선택된 도시 목록.
    selected_channels : list[str] | None
        선택된 채널 목록.
    selected_categories : list[str] | None
        선택된 카테고리 목록.

    Returns
    -------
    pandas.DataFrame
        필터가 적용된 새로운 데이터프레임.
    """
    filtered_frame = source_frame.copy()

    if selected_cities:
        filtered_frame = filtered_frame[filtered_frame["City"].isin(selected_cities)]

    if selected_channels:
        filtered_frame = filtered_frame[filtered_frame["Channel"].isin(selected_channels)]

    if selected_categories:
        filtered_frame = filtered_frame[filtered_frame["Category"].isin(selected_categories)]

    return filtered_frame


def compute_kpis(filtered_frame: pd.DataFrame) -> tuple[float, float, float]:
    """
    KPI(총매출, 총수량, 평균단가)를 계산한다.

    Parameters
    ----------
    filtered_frame : pandas.DataFrame
        필터링된 데이터프레임.

    Returns
    -------
    tuple[float, float, float]
        (total_sales, total_quantity, average_price) 의 튜플.
    """
    total_sales = float(filtered_frame["Sales"].sum()) if "Sales" in filtered_frame.columns else 0.0
    total_quantity = float(filtered_frame["Quantity"].sum()) if "Quantity" in filtered_frame.columns else 0.0
    average_price = float(filtered_frame["AvgPrice_calc"].mean()) if "AvgPrice_calc" in filtered_frame.columns else 0.0
    return total_sales, total_quantity, average_price


def register_callbacks(app: Dash, source_frame: pd.DataFrame) -> None:
    """
    Dash 애플리케이션에 콜백을 등록한다.

    Parameters
    ----------
    app : Dash
        Dash 애플리케이션 인스턴스.
    source_frame : pandas.DataFrame
        원본 데이터프레임(필터링의 기준).
    """

    @app.callback(
        Output("kpi-row", "children"),
        Output("g-category", "figure"),
        Output("g-channel", "figure"),
        Output("g-city", "figure"),
        Output("g-scatter", "figure"),
        Output("g-treemap", "figure"),
        Input("f-city", "value"),
        Input("f-channel", "value"),
        Input("f-category", "value"),
    )
    def update_dashboard(selected_cities: list[str] | None,
                         selected_channels: list[str] | None,
                         selected_categories: list[str] | None):
        """
        드롭다운 선택이 변경될 때마다 전체 대시보드를 갱신한다.

        Parameters
        ----------
        selected_cities : list[str] | None
            City 드롭다운에서 선택된 값.
        selected_channels : list[str] | None
            Channel 드롭다운에서 선택된 값.
        selected_categories : list[str] | None
            Category 드롭다운에서 선택된 값.

        Returns
        -------
        tuple
            KPI 카드 리스트와 5개의 그래프 Figure를 순서대로 반환.
        """
        filtered_frame = apply_filters(
            source_frame,
            selected_cities=selected_cities,
            selected_channels=selected_channels,
            selected_categories=selected_categories,
        )

        total_sales, total_quantity, average_price = compute_kpis(filtered_frame)

        # KPI 카드는 문자열 서식화를 통해 보기 좋게 표현
        kpi_components = [
            kpi_card("Total Sales", f"{total_sales:,.0f}"),
            kpi_card("Total Quantity", f"{total_quantity:,.0f}"),
            kpi_card("Average Price", f"{average_price:,.2f}"),
        ]

        return (
            kpi_components,
            fig_category_sales(filtered_frame),
            fig_channel_share(filtered_frame),
            fig_city_topn(filtered_frame, top_n=10),
            fig_price_qty_scatter(filtered_frame),
            fig_product_treemap(filtered_frame, max_items=50),
        )
