"""
Dash 레이아웃 생성 모듈.

- 필터 드롭다운, KPI 카드, 그래프 영역을 배치한다.
- 레이아웃은 "화면 구성"만 담당하고 데이터 로직은 다루지 않는다.

학습 포인트:
- 레이아웃(UI)과 로직(콜백/계산)을 분리하면 협업과 테스트가 쉬워진다.
"""

from dash import html, dcc
import pandas as pd
from config import APP_STYLES
from services.data_loader import unique_sorted


def kpi_card(title_text: str, value_text: str) -> html.Div:
    """
    간단한 KPI 카드 컴포넌트를 생성한다.

    Parameters
    ----------
    title_text : str
        KPI 제목(예: 'Total Sales').
    value_text : str
        KPI 값(서식 적용된 문자열).

    Returns
    -------
    dash.html.Div
        스타일이 적용된 KPI 카드 컴포넌트.
    """
    return html.Div(
        [
            html.Div(title_text, style={"fontSize": "14px", "color": "#666"}),
            html.Div(value_text, style={"fontSize": "22px", "fontWeight": "600"}),
        ],
        style=APP_STYLES["card"],
    )


def build_layout(data_frame: pd.DataFrame) -> html.Div:
    """
    대시보드 전체 레이아웃을 구성한다.

    Parameters
    ----------
    data_frame : pandas.DataFrame
        드롭다운 옵션 생성을 위한 원본 데이터프레임.

    Returns
    -------
    dash.html.Div
        Dash 애플리케이션의 루트 레이아웃.
    """
    months = unique_sorted(data_frame, "OrderYearMonth")
    regions = unique_sorted(data_frame, "Region")
    categories = unique_sorted(data_frame, "Category")
    segments = unique_sorted(data_frame, "Segment")

    return html.Div(
        [
            html.H2("Superstore Sales 대시보드", style=APP_STYLES["h2"]),

            # Filters Row
            html.Div(
                [
                    html.Div(
                        [
                            html.Label("주문 월"),
                            dcc.Dropdown(
                                id="f-month",
                                options=[{"label": month, "value": month} for month in months],
                                value=None,
                                multi=True,
                                placeholder="연-월을 선택하세요.",
                            ),
                        ],
                        style={"flex": "1", "minWidth": "200px", "marginRight": "12px"},
                    ),
                    html.Div(
                        [
                            html.Label("지역"),
                            dcc.Dropdown(
                                id="f-region",
                                options=[{"label": region, "value": region} for region in regions],
                                value=None,
                                multi=True,
                                placeholder="지역을 선택하세요.",
                            ),
                        ],
                        style={"flex": "1", "minWidth": "200px", "marginRight": "12px"},
                    ),
                    html.Div(
                        [
                            html.Label("카테고리"),
                            dcc.Dropdown(
                                id="f-category",
                                options=[{"label": category, "value": category} for category in categories],
                                value=None,
                                multi=True,
                                placeholder="카테고리를 선택하세요.",
                            ),
                        ],
                        style={"flex": "1", "minWidth": "200px", "marginRight": "12px"},
                    ),
                    html.Div(
                        [
                            html.Label("세그먼트"),
                            dcc.Dropdown(
                                id="f-segment",
                                options=[{"label": segment, "value": segment} for segment in segments],
                                value=None,
                                multi=True,
                                placeholder="세그먼트를 선택하세요.",
                            ),
                        ],
                        style={"flex": "1", "minWidth": "200px"},
                    ),
                ],
                style=APP_STYLES["row"],
            ),

            # KPI Row (콜백으로 채워짐)
            html.Div(id="kpi-row", style=APP_STYLES["row"]),

            # Top Charts Row
            html.Div(
                [
                    html.Div([dcc.Graph(id="g-category")], style={"flex": "1", "minWidth": "360px"}),
                    html.Div([dcc.Graph(id="g-region")], style={"flex": "1", "minWidth": "360px"}),
                ],
                style=APP_STYLES["row"],
            ),

            # Bottom Charts Row
            html.Div(
                [
                    html.Div([dcc.Graph(id="g-state")], style={"flex": "1", "minWidth": "360px"}),
                    html.Div([dcc.Graph(id="g-scatter")], style={"flex": "1", "minWidth": "360px"}),
                    html.Div([dcc.Graph(id="g-treemap")], style={"flex": "1", "minWidth": "360px"}),
                ],
                style=APP_STYLES["row"],
            ),

            html.Div(
                "홍익대학교 소프트웨어융합학과 데이터사이언스 - Superstore Plotly 대시보드 예제",
                style={"color": "#666", "padding": "8px"},
            ),
        ],
        style=APP_STYLES["container"],
    )
