"""
Superstore 데이터 로딩 및 전처리 모듈.

- CSV를 읽어와 숫자/문자형 컬럼을 정규화한다.
- 주문 월(OrderYearMonth)과 주문 건수, 이익률 등의 파생 컬럼을 생성한다.
"""

from pathlib import Path
from typing import Iterable

import pandas as pd


# Superstore 데이터에서 숫자형으로 다루고 싶은 컬럼
NUMERIC_COLUMNS: Iterable[str] = ("Sales", "Profit", "Quantity", "Discount")

# 필터로 활용될 수 있는 텍스트 컬럼
TEXT_COLUMNS: Iterable[str] = ("Region", "Category", "Segment", "Ship Mode", "State", "City", "Sub-Category")


def load_sales_csv(csv_path: Path) -> pd.DataFrame:
    """
    Superstore 원본 CSV를 로드하고 기본 정규화를 수행한다.

    Parameters
    ----------
    csv_path : Path
        CSV 파일 경로.

    Returns
    -------
    pandas.DataFrame
        정규화된 데이터프레임. 핵심 파생 컬럼은 다음과 같다.
        - OrderYearMonth : YYYY-MM 문자열
        - AvgSalesPerItem : Sales / Quantity (Quantity 미존재 시 0)
        - ProfitMargin : Profit / Sales (Profit 또는 Sales가 없으면 0)
    """

    data_frame = pd.read_csv(csv_path)

    # 숫자형 컬럼 정규화
    for column_name in NUMERIC_COLUMNS:
        if column_name in data_frame.columns:
            data_frame[column_name] = pd.to_numeric(data_frame[column_name], errors="coerce").fillna(0)

    # 텍스트 컬럼 결측 처리
    for column_name in TEXT_COLUMNS:
        if column_name in data_frame.columns:
            data_frame[column_name] = data_frame[column_name].fillna("Unknown")

    # 주문일 → 연-월 파생 컬럼
    if "Order Date" in data_frame.columns:
        data_frame["Order Date"] = pd.to_datetime(data_frame["Order Date"], errors="coerce")
        data_frame["OrderYearMonth"] = data_frame["Order Date"].dt.to_period("M").astype(str)
    else:
        data_frame["OrderYearMonth"] = "Unknown"

    # 평균 매출 단가
    if {"Sales", "Quantity"}.issubset(data_frame.columns):
        data_frame["AvgSalesPerItem"] = (
            data_frame["Sales"] / data_frame["Quantity"].replace({0: pd.NA})
        ).fillna(0)
    else:
        data_frame["AvgSalesPerItem"] = 0

    # 이익률
    if {"Sales", "Profit"}.issubset(data_frame.columns):
        data_frame["ProfitMargin"] = (
            data_frame["Profit"] / data_frame["Sales"].replace({0: pd.NA})
        ).fillna(0)
    else:
        data_frame["ProfitMargin"] = 0

    return data_frame


def unique_sorted(data_frame: pd.DataFrame, column_name: str) -> list[str]:
    """
    특정 컬럼의 유니크 문자열 목록을 정렬하여 반환한다.

    Parameters
    ----------
    data_frame : pandas.DataFrame
        원본 데이터프레임.
    column_name : str
        고유값을 추출할 컬럼명.
    """
    if column_name not in data_frame.columns:
        return []
    return sorted(data_frame[column_name].dropna().astype(str).unique().tolist())
