"""
Dash 콜백 관리 모듈.

- 필터 변경 시 KPI와 모든 차트를 갱신한다.
- 데이터 필터링과 KPI 계산을 별도 함수로 분리하여 테스트 용이성을 높인다.

학습 포인트:
- 콜백 함수는 '입력 → 순수 계산 → 출력' 흐름으로 작성하면 디버깅이 쉽다.
"""

import pandas as pd
from dash import Dash, Input, Output
from services.layout import kpi_card
from services.figures import (
    fig_category_sales,
    fig_region_share,
    fig_state_topn,
    fig_monthly_sales,
    fig_product_treemap,
)


def apply_filters(
    source_frame: pd.DataFrame,
    selected_months: list[str] | None,
    selected_regions: list[str] | None,
    selected_categories: list[str] | None,
    selected_segments: list[str] | None,
) -> pd.DataFrame:
    """
    드롭다운에서 선택한 값들로 데이터프레임을 필터링한다.

    Parameters
    ----------
    source_frame : pandas.DataFrame
        원본 데이터프레임.
    selected_months : list[str] | None
        선택된 연-월(OrderYearMonth) 목록.
    selected_regions : list[str] | None
        선택된 지역 목록.
    selected_categories : list[str] | None
        선택된 카테고리 목록.
    selected_segments : list[str] | None
        선택된 고객 세그먼트 목록.

    Returns
    -------
    pandas.DataFrame
        필터가 적용된 새로운 데이터프레임.
    """
    filtered_frame = source_frame.copy()

    if selected_months:
        filtered_frame = filtered_frame[filtered_frame["OrderYearMonth"].isin(selected_months)]

    if selected_regions:
        filtered_frame = filtered_frame[filtered_frame["Region"].isin(selected_regions)]

    if selected_categories:
        filtered_frame = filtered_frame[filtered_frame["Category"].isin(selected_categories)]

    if selected_segments and "Segment" in filtered_frame.columns:
        filtered_frame = filtered_frame[filtered_frame["Segment"].isin(selected_segments)]

    return filtered_frame


def compute_kpis(filtered_frame: pd.DataFrame) -> tuple[float, float, float]:
    """
    KPI(총매출, 주문 수, 평균 주문 금액)를 계산한다.

    Parameters
    ----------
    filtered_frame : pandas.DataFrame
        필터링된 데이터프레임.

    Returns
    -------
    tuple[float, float, float]
        (total_sales, total_quantity, average_price) 의 튜플.
    """
    total_sales = float(filtered_frame["Sales"].sum()) if "Sales" in filtered_frame.columns else 0.0
    total_orders = float(filtered_frame["Order ID"].nunique()) if "Order ID" in filtered_frame.columns else 0.0
    avg_order_value = (total_sales / total_orders) if total_orders else 0.0
    return total_sales, total_orders, avg_order_value


def register_callbacks(app: Dash, source_frame: pd.DataFrame) -> None:
    """
    Dash 애플리케이션에 콜백을 등록한다.

    Parameters
    ----------
    app : Dash
        Dash 애플리케이션 인스턴스.
    source_frame : pandas.DataFrame
        원본 데이터프레임(필터링의 기준).
    """

    @app.callback(
        Output("kpi-row", "children"),
        Output("g-category", "figure"),
        Output("g-region", "figure"),
        Output("g-state", "figure"),
        Output("g-scatter", "figure"),
        Output("g-treemap", "figure"),
        Input("f-month", "value"),
        Input("f-region", "value"),
        Input("f-category", "value"),
        Input("f-segment", "value"),
    )
    def update_dashboard(
        selected_months: list[str] | None,
        selected_regions: list[str] | None,
        selected_categories: list[str] | None,
        selected_segments: list[str] | None,
    ):
        """
        드롭다운 선택이 변경될 때마다 전체 대시보드를 갱신한다.

        Parameters
        ----------
        selected_months : list[str] | None
            OrderYearMonth 드롭다운에서 선택된 값.
        selected_regions : list[str] | None
            Region 드롭다운에서 선택된 값.
        selected_categories : list[str] | None
            Category 드롭다운에서 선택된 값.
        selected_segments : list[str] | None
            Segment 드롭다운에서 선택된 값.

        Returns
        -------
        tuple
            KPI 카드 리스트와 5개의 그래프 Figure를 순서대로 반환.
        """
        filtered_frame = apply_filters(
            source_frame,
            selected_months=selected_months,
            selected_regions=selected_regions,
            selected_categories=selected_categories,
            selected_segments=selected_segments,
        )

        total_sales, total_orders, avg_order_value = compute_kpis(filtered_frame)

        # KPI 카드는 문자열 서식화를 통해 보기 좋게 표현
        kpi_components = [
            kpi_card("총 매출 (Sales)", f"{total_sales:,.0f}"),
            kpi_card("주문 수 (Orders)", f"{total_orders:,.0f}"),
            kpi_card("평균 주문 금액", f"{avg_order_value:,.2f}"),
        ]

        return (
            kpi_components,
            fig_category_sales(filtered_frame),
            fig_region_share(filtered_frame),
            fig_state_topn(filtered_frame, top_n=10),
            fig_monthly_sales(filtered_frame),
            fig_product_treemap(filtered_frame, max_items=50),
        )
